﻿using RimWorld;
using System;                         
using Verse;

namespace MedTimes
{
    public class CompHitpointRefuelable : CompRefuelable
    {
        private CompHeatPusherIce heatPusher;    
                                         
        public CompHitpointRefuelable() : base()
        {
        }

        public override void PostSpawnSetup(bool respawningAfterLoad)
        {
            base.PostSpawnSetup(respawningAfterLoad);
            heatPusher = parent.GetComp<CompHeatPusherIce>();
        }

        public override void ReceiveCompSignal(string signal)
        {
            base.ReceiveCompSignal(signal);

            if (signal == RefueledSignal)
            {
                var max = parent.MaxHitPoints;

                var shouldHaveHp = (int)Math.Ceiling(Fuel / Props.fuelCapacity * max);
                SetHP(shouldHaveHp);
            }
        }

        public override string CompInspectStringExtra()
        {
            var curFuel = Fuel;
            var fuelLevel = (int)Math.Round(curFuel / Props.fuelCapacity * 100);

            var text = $"{Props.FuelLabel}: {fuelLevel}%";

            if (!this.Props.consumeFuelOnlyWhenUsed && this.HasFuel)
            {
                int numTicks = (int)(curFuel / Props.fuelConsumptionRate * 60000f);
                text = text + " (" + numTicks.ToStringTicksToPeriod() + ")";
            }
            if (!this.HasFuel && !this.Props.outOfFuelMessage.NullOrEmpty())
            {
                text += string.Format("\n{0} ({1}x {2})", this.Props.outOfFuelMessage, this.GetFuelCountToFullyRefuel(), this.Props.fuelFilter.AnyAllowedDef.label);
            }
            if (this.Props.targetFuelLevelConfigurable)
            {
                text = text + "\n" + "ConfiguredTargetFuelLevel".Translate(this.TargetFuelLevel.ToStringDecimalIfSmall());
            }
            return text;
        }

        public override void CompTick()
        {
            var max = parent.MaxHitPoints;
            var hp = parent.HitPoints;
            var shouldHaveFuel = (float)hp / max * Props.fuelCapacity;
            if (shouldHaveFuel < Fuel)
            {
                this.ConsumeFuel(Fuel - shouldHaveFuel);
            }                                          

            if (heatPusher == null || heatPusher.PushHeatNow)
                base.CompTick();


            var shouldHaveHp = (int)Math.Ceiling(Fuel / Props.fuelCapacity * max);
            if (shouldHaveHp < hp)
            {
                SetHP(shouldHaveHp);
            }
        }

        private void SetHP(int newHP)
        {
            parent.HitPoints = newHP;
        }
    }
}
